/*
 * Decompiled with CFR 0.152.
 */
package ProteinTools;

import Constants.MolBioConstants;
import ProteinTools.AminoAcid;
import ProteinTools.AminoAcids;
import Sequences.DNA;
import java.beans.BeanProperty;
import java.util.ArrayList;
import java.util.HashMap;

public class CodonTable
implements MolBioConstants {
    protected AminoAcids aminoAcids = new AminoAcids();
    private static AminoAcids defaultAminoAcids = new AminoAcids();
    public static final AminoAcid AA_X = new AminoAcid('X', "Xxx");
    public static final int eFrame_1_Sense = 1;
    public static final int eFrame_1_Antisense = -1;
    public static final int eFrame_2_Sense = 2;
    public static final int eFrame_2_Antisense = -2;
    public static final int eFrame_3_Sense = 3;
    public static final int eFrame_3_Antisense = -3;
    private static final String[] codons = new String[]{"AAA", "AAC", "AAG", "AAT", "ACA", "ACC", "ACG", "ACT", "AGA", "AGC", "AGG", "AGT", "ATA", "ATC", "ATG", "ATT", "CAA", "CAC", "CAG", "CAT", "CCA", "CCC", "CCG", "CCT", "CGA", "CGC", "CGG", "CGT", "CTA", "CTC", "CTG", "CTT", "GAA", "GAC", "GAG", "GAT", "GCA", "GCC", "GCG", "GCT", "GGA", "GGC", "GGG", "GGT", "GTA", "GTC", "GTG", "GTT", "TAA", "TAC", "TAG", "TAT", "TCA", "TCC", "TCG", "TCT", "TGA", "TGC", "TGG", "TGT", "TTA", "TTC", "TTG", "TTT"};
    private static final String[] altStart = new String[]{"1.34", "0.32", "1.00", "NR", "0.17", "0.42", "3.89", "0.13", "0.15", "0.18", "0.91", "0.04", "1.29", "2.71", "49.76", "3.05", "0.07", "0.25", "0.51", "NR", "0.28", "0.16", "0.29", "0.11", "0.12", "0.42", "0.11", "0.07", "0.16", "0.25", "15.44", "0.05", "0.31", "0.31", "0.44", "NR", "0.32", "0.49", "0.38", "0.22", "0.31", "0.39", "0.10", "0.10", "0.12", "0.25", "7.17", "NR", "0.01", "0.23", "<0.01%", "NR", "0.09", "0.13", "0.13", "0.08", "NR", "NR", "NR", "NR", "0.09", "0.29", "4.17", "0.22"};
    private static final String[] orgNames = new String[]{"Human", "Rat", "Mouse", "E.coli", "S.cerevisiae", "Pan troglodytes troglodytes", "Arabidopsis thaliana", "Vertebrate Mitochondria", "Invertebrate Mitochondria", "Eubacterial", "Standard", "Yeast", "Caenorhabditis elegans", "Drosophila melanogaster", "Insect", "Pig", "Streptomyces", "Pichia pastoris", "Chinese Hamster", "Nicotiana tabacum", "Zea mays"};
    private static final String A1 = "A";
    private static final String R1 = "R";
    private static final String N1 = "N";
    private static final String D1 = "D";
    private static final String C1 = "C";
    private static final String E1 = "E";
    private static final String Q1 = "Q";
    private static final String G1 = "G";
    private static final String H1 = "H";
    private static final String I1 = "I";
    private static final String L1 = "L";
    private static final String K1 = "K";
    private static final String M1 = "M";
    private static final String F1 = "F";
    private static final String P1 = "P";
    private static final String S1 = "S";
    private static final String T1 = "T";
    private static final String W1 = "W";
    private static final String Y1 = "Y";
    private static final String V1 = "V";
    private static final String STP1 = "*";
    private static final String A2 = "A  ";
    private static final String R2 = "R  ";
    private static final String N2 = "N  ";
    private static final String D2 = "D  ";
    private static final String C2 = "C  ";
    private static final String E2 = "E  ";
    private static final String Q2 = "Q  ";
    private static final String G2 = "G  ";
    private static final String H2 = "H  ";
    private static final String I2 = "I  ";
    private static final String L2 = "L  ";
    private static final String K2 = "K  ";
    private static final String M2 = "M  ";
    private static final String F2 = "F  ";
    private static final String P2 = "P  ";
    private static final String S2 = "S  ";
    private static final String T2 = "T  ";
    private static final String W2 = "W  ";
    private static final String Y2 = "Y  ";
    private static final String V2 = "V  ";
    private static final String STP2 = "*  ";
    private static final String A3 = "Ala";
    private static final String R3 = "Arg";
    private static final String N3 = "Asn";
    private static final String D3 = "Asp";
    private static final String C3 = "Cys";
    private static final String E3 = "Glu";
    private static final String Q3 = "Gln";
    private static final String G3 = "Gly";
    private static final String H3 = "His";
    private static final String I3 = "Ile";
    private static final String L3 = "Leu";
    private static final String K3 = "Lys";
    private static final String M3 = "Met";
    private static final String F3 = "Phe";
    private static final String P3 = "Pro";
    private static final String S3 = "Ser";
    private static final String T3 = "Thr";
    private static final String W3 = "Trp";
    private static final String Y3 = "Tyr";
    private static final String V3 = "Val";
    private static int organismCount;
    private static final String[][] A;
    private static final String[][] AA;
    private static final String[][] AAA;
    private static final char[][] cAA;
    private static final int[][] frequencies;
    private static LetterCode letterCode;
    private static Organism[] organisms;
    private static HashMap<String, Organism> mappedOrganisms;
    private Organism organism = organisms[0];

    public AminoAcids getAminoAcids() {
        return this.aminoAcids;
    }

    public static int getOrganismIndex(String orgName) {
        if (!mappedOrganisms.containsKey(orgName.toLowerCase())) {
            return -1;
        }
        return CodonTable.mappedOrganisms.get((Object)orgName.toLowerCase()).index;
    }

    public static int getOrganismCount() {
        return organismCount;
    }

    public static void setCodonFormat(LetterCode letterCode) {
        CodonTable.letterCode = letterCode;
    }

    public static LetterCode getCodonFormat() {
        return letterCode;
    }

    public static String[] getCodonsForAA(char codon, int orgIndex) {
        return orgIndex >= 0 && orgIndex < organisms.length ? CodonTable.getCodonsForAA(codon, organisms[orgIndex]) : null;
    }

    public static String[] getCodonsForAA(char codon, Organism organism) {
        ArrayList<String> sb = new ArrayList<String>();
        for (int x = 0; x < codons.length; ++x) {
            if (organism.cAA[x] != codon) continue;
            sb.add(codons[x]);
        }
        if (sb.isEmpty()) {
            return null;
        }
        String[] outVal = new String[sb.size()];
        for (int x = 0; x < sb.size(); ++x) {
            outVal[x] = (String)sb.get(x);
        }
        return outVal;
    }

    public static int getCodonFrequency(String codon, int orgNum) {
        if (codon.length() == 3) {
            return CodonTable.getCodonFrequency(codon.charAt(0), codon.charAt(1), codon.charAt(2), orgNum);
        }
        return 0;
    }

    public static int getCodonFrequency(String codon, Organism organism) {
        if (codon.length() == 3) {
            return CodonTable.getCodonFrequency(codon.charAt(0), codon.charAt(1), codon.charAt(2), organism);
        }
        return 0;
    }

    public static int getCodonFrequency(char c1, char c2, char c3, int orgNum) {
        return orgNum >= 0 && orgNum < organisms.length ? CodonTable.getCodonFrequency(c1, c2, c3, organisms[orgNum]) : 0;
    }

    public static int getCodonFrequency(char c1, char c2, char c3, Organism organism) {
        int cIndex = CodonTable.getCodonIndex(c1, c2, c3);
        if (cIndex == -1) {
            return 0;
        }
        return organism.frequencies[cIndex];
    }

    public static String getNearestCodon(String newAA, String OldCodon, int orgTableIndex) {
        int highestMatchIndex;
        block7: {
            int highestMatch;
            block6: {
                highestMatch = 0;
                highestMatchIndex = -1;
                if (newAA.length() != 1) break block6;
                for (int x = 0; x < codons.length; ++x) {
                    if (!AA[orgTableIndex][x].equalsIgnoreCase(newAA)) continue;
                    int cMatch = 0;
                    if (highestMatchIndex == -1) {
                        highestMatchIndex = x;
                    }
                    for (int y = 0; y < 3; ++y) {
                        if (codons[x].charAt(y) != OldCodon.charAt(y)) continue;
                        ++cMatch;
                    }
                    if (cMatch <= highestMatch) continue;
                    highestMatch = cMatch;
                    highestMatchIndex = x;
                }
                break block7;
            }
            if (newAA.length() != 3) break block7;
            for (int x = 0; x < codons.length; ++x) {
                if (!AAA[orgTableIndex][x].equalsIgnoreCase(newAA)) continue;
                if (highestMatchIndex == -1) {
                    highestMatchIndex = x;
                }
                int cMatch = 0;
                for (int y = 0; y < 3; ++y) {
                    if (codons[x].charAt(y) != OldCodon.charAt(y)) continue;
                    ++cMatch;
                }
                if (cMatch <= highestMatch) continue;
                highestMatch = cMatch;
                highestMatchIndex = x;
            }
        }
        return codons[highestMatchIndex];
    }

    public static String getNearestCodon(String newAA, String OldCodon, Organism organism) {
        int highestMatchIndex;
        block7: {
            int highestMatch;
            block6: {
                highestMatch = 0;
                highestMatchIndex = -1;
                if (newAA.length() != 1) break block6;
                for (int x = 0; x < codons.length; ++x) {
                    if (!organism.AA[x].equalsIgnoreCase(newAA)) continue;
                    int cMatch = 0;
                    if (highestMatchIndex == -1) {
                        highestMatchIndex = x;
                    }
                    for (int y = 0; y < 3; ++y) {
                        if (codons[x].charAt(y) != OldCodon.charAt(y)) continue;
                        ++cMatch;
                    }
                    if (cMatch <= highestMatch) continue;
                    highestMatch = cMatch;
                    highestMatchIndex = x;
                }
                break block7;
            }
            if (newAA.length() != 3) break block7;
            for (int x = 0; x < codons.length; ++x) {
                if (!organism.AAA[x].equalsIgnoreCase(newAA)) continue;
                if (highestMatchIndex == -1) {
                    highestMatchIndex = x;
                }
                int cMatch = 0;
                for (int y = 0; y < 3; ++y) {
                    if (codons[x].charAt(y) != OldCodon.charAt(y)) continue;
                    ++cMatch;
                }
                if (cMatch <= highestMatch) continue;
                highestMatch = cMatch;
                highestMatchIndex = x;
            }
        }
        return codons[highestMatchIndex];
    }

    public static int getCodonIndex(char c1, char c2, char c3) {
        int cIndex = 0;
        switch (c1) {
            case 'A': 
            case 'a': {
                break;
            }
            case 'C': 
            case 'c': {
                cIndex = 16;
                break;
            }
            case 'G': 
            case 'g': {
                cIndex = 32;
                break;
            }
            case 'T': 
            case 'U': 
            case 't': 
            case 'u': {
                cIndex = 48;
                break;
            }
            default: {
                return -1;
            }
        }
        switch (c2) {
            case 'A': 
            case 'a': {
                break;
            }
            case 'C': 
            case 'c': {
                cIndex += 4;
                break;
            }
            case 'G': 
            case 'g': {
                cIndex += 8;
                break;
            }
            case 'T': 
            case 'U': 
            case 't': 
            case 'u': {
                cIndex += 12;
                break;
            }
            default: {
                return -1;
            }
        }
        switch (c3) {
            case 'A': 
            case 'a': {
                break;
            }
            case 'C': 
            case 'c': {
                ++cIndex;
                break;
            }
            case 'G': 
            case 'g': {
                cIndex += 2;
                break;
            }
            case 'T': 
            case 'U': 
            case 't': 
            case 'u': {
                cIndex += 3;
                break;
            }
            default: {
                return -1;
            }
        }
        return cIndex;
    }

    public static AminoAcid translateCodon(char c1, char c2, char c3, int orgIndex) {
        return orgIndex >= 0 && orgIndex < organisms.length ? CodonTable.translateCodon(c1, c2, c3, organisms[orgIndex]) : CodonTable.defaultAminoAcids.nullAA;
    }

    public static AminoAcid translateCodon(char c1, char c2, char c3, Organism organism) {
        int cIndex = CodonTable.getCodonIndex(c1, c2, c3);
        if (cIndex == -1) {
            return CodonTable.defaultAminoAcids.nullAA;
        }
        return defaultAminoAcids.getAA(organism.cAA[cIndex]);
    }

    @BeanProperty(hidden=true)
    public static String getOrganismName(int index) {
        return index >= 0 && index < organisms.length ? CodonTable.organisms[index].name : "";
    }

    public static String filterAA(String Seq) {
        return Seq.replaceAll("[^*ACDEFGHIKLMNPQRSTVWYacdefghiklmnpqrstvwy]", "");
    }

    public static boolean isAA(String Seq) {
        return Seq.length() == Seq.replaceAll("[^*ACDEFGHIKLMNPQRSTVWYacdefghiklmnpqrstvwy]", "").length();
    }

    public static String AAtoRegex(String Seq) {
        if (Seq.length() == 0) {
            return "";
        }
        return Seq.toUpperCase().replaceAll("X", "[ACDEFGHIKLMNOPQRSTUVWY]").replaceAll("B", "(D|N)").replaceAll("Z", "(E|Q)");
    }

    private static Organism addOrganism(String name, int[] frequencyTable, int tableIndex, int index) {
        String lName = name.toLowerCase();
        if (mappedOrganisms.containsKey(lName)) {
            return null;
        }
        Organism org = new Organism();
        org.name = name;
        org.frequencies = frequencyTable;
        org.cAA = cAA[tableIndex];
        org.A = A[tableIndex];
        org.AA = AA[tableIndex];
        org.AAA = AAA[tableIndex];
        org.isDefault = true;
        org.index = index;
        mappedOrganisms.put(lName, org);
        return org;
    }

    public static boolean addOrganism(String name, int[] frequencyTable, char[] aminoAcids) {
        String lName = name.toLowerCase();
        if (mappedOrganisms.containsKey(lName)) {
            return false;
        }
        Organism org = new Organism();
        org.name = name;
        org.frequencies = frequencyTable;
        org.cAA = aminoAcids;
        int size = org.cAA.length;
        org.A = new String[size];
        org.AA = new String[size];
        org.AAA = new String[size];
        for (int x = 0; x < size; ++x) {
            AminoAcid aa = defaultAminoAcids.getAA(org.cAA[x]);
            org.A[x] = "" + aa.abbr1;
            org.AA[x] = aa.abbr1 + "  ";
            org.AAA[x] = aa.abbr3;
        }
        mappedOrganisms.put(lName, org);
        return true;
    }

    public String getHighestCodon(char aa) {
        if (aa > '`') {
            aa = (char)(aa - 32);
        }
        int highestFreq = -1;
        int highestFreqIndex = -1;
        for (int x = 0; x < codons.length; ++x) {
            if (this.organism.cAA[x] != aa || this.organism.frequencies[x] <= highestFreq) continue;
            highestFreq = this.organism.frequencies[x];
            highestFreqIndex = x;
        }
        if (highestFreqIndex > -1) {
            return codons[highestFreqIndex];
        }
        return "";
    }

    public String getHighestCodon(String aminoAcid) {
        return this.getHighestCodon(aminoAcid, this.organism);
    }

    public String getHighestCodon(String aminoAcid, Organism organism) {
        int highestFreq = -1;
        int highestFreqIndex = -1;
        if (aminoAcid.length() == 1) {
            return this.getHighestCodon(aminoAcid.charAt(0));
        }
        if (aminoAcid.length() == 3) {
            for (int x = 0; x < codons.length; ++x) {
                if (!organism.AAA[x].equalsIgnoreCase(aminoAcid) || organism.frequencies[x] <= highestFreq) continue;
                highestFreq = organism.frequencies[x];
                highestFreqIndex = x;
            }
        }
        return codons[highestFreqIndex];
    }

    public String translate(String Seq) {
        return this.pTranslateSeq(DNA.filterSequence(Seq), 1, letterCode, false);
    }

    public String translate(String Seq, LetterCode returnFormat) {
        return this.pTranslateSeq(DNA.filterSequence(Seq), 1, returnFormat, false);
    }

    public String translate(String Seq, boolean stopAtTerm) {
        return this.pTranslateSeq(DNA.filterSequence(Seq), 1, letterCode, stopAtTerm);
    }

    public String translate(String Seq, LetterCode returnFormat, boolean stopAtTerm) {
        return this.pTranslateSeq(DNA.filterSequence(Seq), 1, returnFormat, stopAtTerm);
    }

    public String translate(String Seq, int rFrame) {
        return this.pTranslateSeq(DNA.filterSequence(Seq), rFrame, letterCode, false);
    }

    public String translate(String Seq, int bpStart, int bpStop, int rFrame) {
        return this.pTranslateSeq(DNA.filterSequence(Seq.substring(bpStart - 1, bpStop)), rFrame, letterCode, false);
    }

    public String translate(DNA dna, int bpStart, int bpStop, int rFrame) {
        return this.pTranslateSeq(dna.getSequence(bpStart, bpStop), rFrame, letterCode, false);
    }

    public String translate(DNA dna, int rFrame) {
        return this.pTranslateSeq(dna.getSequence(), rFrame, letterCode, false);
    }

    public String translate(String Seq, int rFrame, LetterCode returnFormat) {
        return this.pTranslateSeq(DNA.filterSequence(Seq), rFrame, returnFormat, false);
    }

    public String translate(String Seq, int bpStart, int bpStop, int rFrame, LetterCode returnFormat) {
        return this.pTranslateSeq(DNA.filterSequence(Seq.substring(bpStart - 1, bpStop)), rFrame, returnFormat, false);
    }

    public String translate(DNA dna, int bpStart, int bpStop, int rFrame, LetterCode returnFormat) {
        return this.pTranslateSeq(dna.getSequence(bpStart, bpStop), rFrame, returnFormat, false);
    }

    public String translate(DNA dna, int rFrame, LetterCode returnFormat) {
        return this.pTranslateSeq(dna.getSequence(), rFrame, returnFormat, false);
    }

    public String translate(String Seq, int rFrame, boolean stopAtTerm) {
        return this.pTranslateSeq(DNA.filterSequence(Seq), rFrame, letterCode, stopAtTerm);
    }

    public String translate(String Seq, int bpStart, int bpStop, int rFrame, boolean stopAtTerm) {
        return this.pTranslateSeq(DNA.filterSequence(Seq.substring(bpStart - 1, bpStop)), rFrame, letterCode, stopAtTerm);
    }

    public String translate(DNA dna, int bpStart, int bpStop, int rFrame, boolean stopAtTerm) {
        return this.pTranslateSeq(dna.getSequence(bpStart, bpStop), rFrame, letterCode, stopAtTerm);
    }

    public String translate(DNA dna, int rFrame, boolean stopAtTerm) {
        return this.pTranslateSeq(dna.getSequence(), rFrame, letterCode, stopAtTerm);
    }

    public String translate(String Seq, int rFrame, LetterCode returnFormat, boolean stopAtTerm) {
        return this.pTranslateSeq(DNA.filterSequence(Seq), rFrame, returnFormat, stopAtTerm);
    }

    public String translate(String Seq, int bpStart, int bpStop, int rFrame, LetterCode returnFormat, boolean stopAtTerm) {
        return this.pTranslateSeq(DNA.filterSequence(Seq.substring(bpStart - 1, bpStop)), rFrame, returnFormat, stopAtTerm);
    }

    public String translate(DNA dna, int bpStart, int bpStop, int rFrame, LetterCode returnFormat, boolean stopAtTerm) {
        return this.pTranslateSeq(dna.getSequence(bpStart, bpStop), rFrame, returnFormat, stopAtTerm);
    }

    public String translate(DNA dna, int rFrame, LetterCode returnFormat, boolean stopAtTerm) {
        return this.pTranslateSeq(dna.getSequence(), rFrame, returnFormat, stopAtTerm);
    }

    public String[] getCodonsForAA(char codon) {
        return CodonTable.getCodonsForAA(codon, this.organism);
    }

    public int getCodonFrequency(String codon) {
        return CodonTable.getCodonFrequency(codon, this.organism);
    }

    public int getCodonFrequency(char c1, char c2, char c3) {
        return CodonTable.getCodonFrequency(c1, c2, c3, this.organism);
    }

    public String getNearestCodon(String newAA, String OldCodon) {
        return CodonTable.getNearestCodon(newAA, OldCodon, this.organism);
    }

    public AminoAcid translateCodon(char c1, char c2, char c3) {
        int cIndex = CodonTable.getCodonIndex(c1, c2, c3);
        if (cIndex == -1) {
            return null;
        }
        return this.aminoAcids.getAA(this.organism.cAA[cIndex]);
    }

    public AminoAcid translateCodonX(char c1, char c2, char c3) {
        AminoAcid val = this.translateCodon(c1, c2, c3);
        if (val == null) {
            val = AA_X;
        }
        return val;
    }

    private String pTranslateSeq(String Seq, int rFrame, LetterCode returnFormat, boolean stopAtTerm) {
        String tString = null;
        switch (rFrame) {
            case -3: 
            case -2: 
            case -1: {
                tString = DNA.getAntisense(Seq);
                break;
            }
            case 1: 
            case 2: 
            case 3: {
                tString = Seq;
            }
        }
        switch (rFrame) {
            case -2: 
            case 2: {
                tString = tString.substring(1);
                break;
            }
            case -3: 
            case 3: {
                tString = tString.substring(2);
            }
        }
        switch (tString.length() % 3) {
            case 0: {
                break;
            }
            case 1: {
                tString = tString + "  ";
                break;
            }
            case 2: {
                tString = tString + " ";
            }
        }
        StringBuilder OutString = new StringBuilder();
        char[] chars = tString.toCharArray();
        int end = tString.length() - 2;
        for (int codonIndex = 0; codonIndex < end; codonIndex += 3) {
            String aa = this.translateCodon(chars[codonIndex], chars[codonIndex + 1], chars[codonIndex + 2], returnFormat);
            OutString.append(aa);
            if (!stopAtTerm || aa != STP1 && aa != STP2) continue;
            return OutString.toString();
        }
        return OutString.toString();
    }

    public String translate(char[] sequence, int start, LetterCode returnFormat, boolean stopAtTerm) {
        StringBuilder OutString = new StringBuilder();
        for (int codonIndex = start; codonIndex < sequence.length; codonIndex += 3) {
            String aa = this.translateCodon(sequence[codonIndex], sequence[codonIndex + 1], sequence[codonIndex + 2], returnFormat);
            OutString.append(aa);
            if (!stopAtTerm || aa != STP1 && aa != STP2) continue;
            return OutString.toString();
        }
        return OutString.toString();
    }

    public String translateCodon(String Codon2, LetterCode returnFormat) {
        return this.translateCodon(Codon2.charAt(0), Codon2.charAt(1), Codon2.charAt(2), returnFormat);
    }

    public String translateCodon(char nt1, char nt2, char nt3, LetterCode returnFormat) {
        int cIndex = CodonTable.getCodonIndex(nt1, nt2, nt3);
        if (cIndex == -1) {
            return "X";
        }
        switch (returnFormat) {
            case eSpaced_1_Letter_Code: {
                return this.organism.A[cIndex];
            }
            case e1_Letter_Code: {
                return this.organism.AA[cIndex];
            }
        }
        return this.organism.AAA[cIndex];
    }

    public String getOrganismName() {
        return this.organism.name;
    }

    public int getOrganism() {
        return this.organism.index;
    }

    @BeanProperty(hidden=true, description="Sets the organism with the specified name as the current.")
    public void setOrganism(String name) {
        if (mappedOrganisms.containsKey(name.toLowerCase())) {
            this.organism = mappedOrganisms.get(name.toLowerCase());
        }
    }

    public int getOrganismIndex() {
        return this.organism.index;
    }

    @BeanProperty(hidden=true, description="Sets the organism at the specified index as the current.")
    public void setOrganism(int orgNum) {
        if (orgNum >= 0 && orgNum < organisms.length) {
            this.organism = organisms[orgNum];
        }
    }

    @BeanProperty(hidden=true, description="Converts an amino acid string to a DNA string using the highest scoring codons of the specified organism.")
    public String AA_To_DNA(String Seq, int orgIndex) {
        return this.AA_To_DNA(Seq, organisms[orgIndex]);
    }

    @BeanProperty(hidden=true, description="Converts an amino acid string to a DNA string using the highest scoring codons of the currently set organism.")
    public String AA_To_DNA(String seq) {
        return this.AA_To_DNA(seq, this.organism);
    }

    @BeanProperty(hidden=true, description="Converts an amino acid string to a DNA string using the highest scoring codons of the specified organism.")
    public String AA_To_DNA(String seq, Organism organism) {
        String tmpSeq = CodonTable.filterAA(seq);
        StringBuilder OutSeq = new StringBuilder();
        if (tmpSeq.length() == 0) {
            return "";
        }
        for (int x = 0; x < tmpSeq.length(); ++x) {
            OutSeq.append(this.getHighestCodon(tmpSeq.substring(x, x + 1), organism));
        }
        return OutSeq.toString();
    }

    static {
        A = new String[][]{{K2, N2, K2, N2, T2, T2, T2, T2, R2, S2, R2, S2, I2, I2, M2, I2, Q2, H2, Q2, H2, P2, P2, P2, P2, R2, R2, R2, R2, L2, L2, L2, L2, E2, D2, E2, D2, A2, A2, A2, A2, G2, G2, G2, G2, V2, V2, V2, V2, STP2, Y2, STP2, Y2, S2, S2, S2, S2, STP2, C2, W2, C2, L2, F2, L2, F2}, {K2, M2, K2, N2, T2, T2, T2, V2, Q2, S2, R2, S2, H2, I2, M2, I2, P2, G2, Q2, H2, N2, P2, P2, P2, R2, R2, R2, S2, L2, L2, L2, L2, D2, C2, E2, D2, STP2, A2, A2, A2, F2, G2, G2, G2, V2, V2, V2, W2, STP2, Y2, STP2, Y2, S2, S2, S2, T2, STP2, A2, W2, C2, L2, E2, L2, F2}, {K2, N2, K2, N2, T2, T2, T2, T2, R2, S2, R2, S2, M2, I2, M2, I2, Q2, H2, Q2, H2, P2, P2, P2, P2, R2, R2, R2, R2, L2, L2, L2, L2, E2, D2, E2, D2, A2, A2, A2, A2, G2, G2, G2, G2, V2, V2, V2, V2, STP2, Y2, STP2, Y2, S2, S2, S2, S2, Y2, C2, W2, C2, L2, F2, L2, F2}, {K2, N2, K2, N2, T2, T2, T2, T2, R2, S2, R2, S2, I2, I2, M2, I2, Q2, H2, Q2, H2, P2, P2, P2, P2, R2, R2, R2, R2, L2, L2, S2, L2, E2, D2, E2, D2, A2, A2, A2, A2, G2, G2, G2, G2, V2, V2, V2, V2, STP2, Y2, STP2, Y2, S2, S2, S2, S2, STP2, C2, W2, C2, L2, F2, L2, F2}};
        AA = new String[][]{{K1, N1, K1, N1, T1, T1, T1, T1, R1, S1, R1, S1, I1, I1, M1, I1, Q1, H1, Q1, H1, P1, P1, P1, P1, R1, R1, R1, R1, L1, L1, L1, L1, E1, D1, E1, D1, A1, A1, A1, A1, G1, G1, G1, G1, V1, V1, V1, V1, STP1, Y1, STP1, Y1, S1, S1, S1, S1, STP1, C1, W1, C1, L1, F1, L1, F1}, {K1, M1, K1, N1, T1, T1, T1, V1, Q1, S1, R1, S1, H1, I1, M1, I1, P1, G1, Q1, H1, N1, P1, P1, P1, R1, R1, R1, S1, L1, L1, L1, L1, D1, C1, E1, D1, STP1, A1, A1, A1, F1, G1, G1, G1, V1, V1, V1, W1, STP1, Y1, STP1, Y1, S1, S1, S1, T1, STP1, A1, W1, C1, L1, E1, L1, F1}, {K1, N1, K1, N1, T1, T1, T1, T1, R1, S1, R1, S1, M1, I1, M1, I1, Q1, H1, Q1, H1, P1, P1, P1, P1, R1, R1, R1, R1, L1, L1, L1, L1, E1, D1, E1, D1, A1, A1, A1, A1, G1, G1, G1, G1, V1, V1, V1, V1, STP1, Y1, STP1, Y1, S1, S1, S1, S1, Y1, C1, W1, C1, L1, F1, L1, F1}, {K1, N1, K1, N1, T1, T1, T1, T1, R1, S1, R1, S1, I1, I1, M1, I1, Q1, H1, Q1, H1, P1, P1, P1, P1, R1, R1, R1, R1, L1, L1, S1, L1, E1, D1, E1, D1, A1, A1, A1, A1, G1, G1, G1, G1, V1, V1, V1, V1, STP1, Y1, STP1, Y1, S1, S1, S1, S1, STP1, C1, W1, C1, L1, F1, L1, F1}};
        AAA = new String[][]{{K3, N3, K3, N3, T3, T3, T3, T3, R3, S3, R3, S3, I3, I3, M3, I3, Q3, H3, Q3, H3, P3, P3, P3, P3, R3, R3, R3, R3, L3, L3, L3, L3, E3, D3, E3, D3, A3, A3, A3, A3, G3, G3, G3, G3, V3, V3, V3, V3, STP2, Y3, STP2, Y3, S3, S3, S3, S3, STP2, C3, W3, C3, L3, F3, L3, F3}, {K3, M3, K3, N3, T3, T3, T3, V3, Q3, S3, R3, S3, H3, I3, M3, I3, P3, G3, Q3, H3, N3, P3, P3, P3, R3, R3, R3, S3, L3, L3, L3, L3, D3, C3, E3, D3, STP2, A3, A3, A3, F3, G3, G3, G3, V3, V3, V3, W3, STP2, Y3, STP2, Y3, S3, S3, S3, T3, STP2, A3, W3, C3, L3, E3, L3, F3}, {K3, N3, K3, N3, T3, T3, T3, T3, R3, S3, R3, S3, M3, I3, M3, I3, Q3, H3, Q3, H3, P3, P3, P3, P3, R3, R3, R3, R3, L3, L3, L3, L3, E3, D3, E3, D3, A3, A3, A3, A3, G3, G3, G3, G3, V3, V3, V3, V3, STP2, Y3, STP2, Y3, S3, S3, S3, S3, Y3, C3, W3, C3, L3, F3, L3, F3}, {K3, N3, K3, N3, T3, T3, T3, T3, R3, S3, R3, S3, I3, I3, M3, I3, Q3, H3, Q3, H3, P3, P3, P3, P3, R3, R3, R3, R3, L3, L3, S3, L3, E3, D3, E3, D3, A3, A3, A3, A3, G3, G3, G3, G3, V3, V3, V3, V3, STP2, Y3, STP2, Y3, S3, S3, S3, S3, STP2, C3, W3, C3, L3, F3, L3, F3}};
        cAA = new char[][]{{'K', 'N', 'K', 'N', 'T', 'T', 'T', 'T', 'R', 'S', 'R', 'S', 'I', 'I', 'M', 'I', 'Q', 'H', 'Q', 'H', 'P', 'P', 'P', 'P', 'R', 'R', 'R', 'R', 'L', 'L', 'L', 'L', 'E', 'D', 'E', 'D', 'A', 'A', 'A', 'A', 'G', 'G', 'G', 'G', 'V', 'V', 'V', 'V', '*', 'Y', '*', 'Y', 'S', 'S', 'S', 'S', '*', 'C', 'W', 'C', 'L', 'F', 'L', 'F'}, {'K', 'M', 'K', 'N', 'T', 'T', 'T', 'V', 'Q', 'S', 'R', 'S', 'H', 'I', 'M', 'I', 'P', 'G', 'Q', 'H', 'N', 'P', 'P', 'P', 'R', 'R', 'R', 'S', 'L', 'L', 'L', 'L', 'D', 'C', 'E', 'D', '*', 'A', 'A', 'A', 'F', 'G', 'G', 'G', 'V', 'V', 'V', 'W', '*', 'Y', '*', 'Y', 'S', 'S', 'S', 'T', '*', 'A', 'W', 'C', 'L', 'E', 'L', 'F'}, {'K', 'N', 'K', 'N', 'T', 'T', 'T', 'T', 'R', 'S', 'R', 'S', 'M', 'I', 'M', 'I', 'Q', 'H', 'Q', 'H', 'P', 'P', 'P', 'P', 'R', 'R', 'R', 'R', 'L', 'L', 'L', 'L', 'E', 'D', 'E', 'D', 'A', 'A', 'A', 'A', 'G', 'G', 'G', 'G', 'V', 'V', 'V', 'V', '*', 'Y', '*', 'Y', 'S', 'S', 'S', 'S', 'Y', 'C', 'W', 'C', 'L', 'F', 'L', 'F'}, {'K', 'N', 'K', 'N', 'T', 'T', 'T', 'T', 'R', 'S', 'R', 'S', 'I', 'I', 'M', 'I', 'Q', 'H', 'Q', 'H', 'P', 'P', 'P', 'P', 'R', 'R', 'R', 'R', 'L', 'L', 'S', 'L', 'E', 'D', 'E', 'D', 'A', 'A', 'A', 'A', 'G', 'G', 'G', 'G', 'V', 'V', 'V', 'V', '*', 'Y', '*', 'Y', 'S', 'S', 'S', 'S', '*', 'C', 'W', 'C', 'L', 'F', 'L', 'F'}};
        frequencies = new int[][]{{42, 54, 58, 46, 28, 36, 12, 24, 20, 24, 20, 15, 16, 48, 100, 36, 25, 59, 75, 41, 27, 33, 11, 28, 11, 19, 21, 8, 7, 20, 41, 13, 42, 54, 58, 46, 23, 40, 11, 26, 25, 34, 25, 16, 11, 24, 47, 18, 28, 57, 20, 43, 15, 22, 6, 18, 52, 55, 100, 45, 7, 55, 13, 45}, {37, 60, 63, 40, 28, 37, 12, 23, 20, 25, 21, 15, 14, 54, 100, 32, 25, 62, 75, 38, 27, 32, 11, 30, 12, 18, 20, 9, 7, 21, 42, 12, 39, 58, 61, 42, 22, 40, 10, 28, 25, 34, 24, 17, 11, 26, 47, 16, 27, 60, 23, 40, 14, 23, 6, 18, 50, 56, 100, 45, 6, 59, 12, 41}, {39, 57, 61, 43, 29, 35, 11, 25, 21, 24, 22, 15, 16, 50, 100, 34, 25, 60, 75, 40, 28, 31, 10, 30, 12, 18, 19, 9, 8, 20, 39, 13, 40, 56, 60, 44, 23, 38, 10, 29, 26, 33, 23, 18, 12, 25, 46, 17, 26, 58, 22, 43, 14, 22, 5, 19, 52, 52, 100, 48, 6, 57, 13, 43}, {74, 51, 26, 49, 17, 40, 25, 19, 7, 25, 4, 16, 11, 39, 100, 49, 34, 43, 66, 57, 20, 13, 49, 18, 7, 36, 11, 36, 4, 10, 47, 12, 68, 37, 32, 63, 23, 26, 33, 18, 13, 37, 15, 35, 17, 20, 35, 28, 61, 41, 9, 59, 14, 15, 14, 17, 30, 54, 100, 46, 14, 42, 13, 58}, {58, 41, 42, 59, 31, 22, 14, 34, 48, 11, 20, 16, 28, 26, 100, 46, 69, 36, 31, 64, 41, 16, 11, 32, 7, 7, 5, 14, 14, 5, 12, 13, 71, 34, 29, 66, 29, 23, 11, 38, 22, 20, 12, 47, 21, 21, 19, 39, 33, 44, 33, 56, 21, 16, 10, 27, 33, 38, 100, 62, 28, 41, 29, 59}, {57, 40, 43, 60, 36, 25, 7, 32, 41, 22, 24, 14, 26, 33, 100, 41, 49, 37, 51, 63, 50, 19, 4, 27, 11, 7, 8, 9, 8, 19, 27, 22, 60, 47, 40, 53, 26, 37, 7, 30, 29, 23, 20, 28, 17, 22, 37, 23, 39, 46, 18, 54, 21, 17, 5, 21, 43, 61, 100, 39, 8, 40, 14, 60}, {49, 48, 51, 52, 31, 20, 15, 34, 35, 13, 20, 16, 24, 35, 100, 41, 56, 39, 44, 61, 33, 11, 18, 38, 12, 7, 9, 17, 11, 17, 11, 26, 52, 32, 48, 68, 27, 16, 14, 43, 37, 14, 16, 34, 15, 19, 26, 40, 36, 48, 20, 52, 20, 13, 10, 28, 44, 40, 100, 60, 14, 49, 22, 51}, {49, 66, 51, 48, 20, 15, 34, 15, 44, 16, 26, 20, 61, 46, 34, 54, 38, 34, 56, 39, 52, 33, 11, 18, 16, 20, 38, 13, 11, 17, 11, 26, 68, 60, 52, 32, 35, 27, 16, 14, 51, 37, 14, 16, 19, 26, 40, 9, 3, 48, 2, 52, 13, 10, 28, 31, 60, 43, 91, 40, 14, 48, 22, 49}, {49, 48, 51, 52, 31, 20, 15, 34, 35, 13, 20, 16, 34, 35, 66, 41, 56, 39, 44, 61, 33, 11, 18, 38, 12, 7, 9, 17, 11, 17, 11, 26, 52, 32, 48, 68, 27, 16, 14, 43, 37, 14, 16, 34, 15, 19, 26, 40, 64, 46, 36, 50, 20, 13, 10, 28, 4, 40, 100, 60, 14, 49, 22, 51}, {89, 60, 11, 40, 34, 27, 0, 39, 11, 9, 0, 4, 26, 39, 100, 35, 100, 50, 0, 50, 32, 22, 2, 44, 56, 11, 11, 11, 24, 21, 4, 28, 67, 62, 33, 38, 30, 22, 0, 47, 34, 34, 12, 20, 52, 13, 5, 30, 0, 36, 0, 64, 30, 24, 0, 33, 100, 75, 100, 25, 24, 59, 0, 41}, {89, 60, 11, 40, 34, 27, 0, 39, 11, 9, 0, 4, 26, 39, 100, 35, 100, 50, 0, 50, 32, 22, 2, 44, 56, 11, 11, 11, 24, 21, 4, 28, 67, 62, 33, 38, 30, 22, 0, 47, 34, 34, 12, 20, 52, 13, 5, 30, 0, 36, 0, 64, 30, 24, 0, 33, 100, 75, 100, 25, 24, 59, 0, 41}, {67, 32, 33, 68, 33, 18, 9, 40, 30, 7, 12, 17, 31, 21, 100, 47, 72, 27, 28, 73, 29, 16, 13, 42, 20, 7, 8, 22, 15, 10, 9, 24, 68, 25, 32, 75, 27, 16, 13, 44, 41, 10, 16, 32, 28, 14, 20, 37, 51, 26, 18, 74, 19, 15, 8, 26, 31, 31, 100, 69, 31, 35, 21, 65}, {59, 38, 41, 62, 34, 18, 15, 33, 29, 10, 8, 15, 16, 31, 100, 53, 66, 39, 34, 61, 53, 9, 20, 18, 23, 10, 9, 21, 9, 17, 14, 24, 62, 33, 38, 68, 31, 20, 13, 36, 59, 12, 8, 20, 16, 22, 23, 39, 44, 44, 17, 56, 25, 13, 15, 21, 39, 45, 100, 55, 12, 50, 23, 50}, {29, 56, 71, 44, 19, 38, 26, 17, 9, 25, 11, 14, 19, 47, 100, 34, 30, 60, 70, 40, 25, 33, 29, 13, 15, 33, 15, 16, 9, 15, 43, 10, 33, 47, 67, 53, 17, 45, 19, 19, 29, 43, 7, 21, 11, 24, 47, 18, 42, 63, 32, 37, 9, 24, 20, 8, 26, 71, 100, 29, 5, 63, 18, 37}, {31, 71, 69, 29, 21, 36, 16, 27, 16, 17, 21, 12, 12, 59, 100, 29, 39, 68, 61, 32, 24, 31, 16, 29, 8, 24, 5, 25, 8, 22, 31, 13, 41, 63, 59, 37, 16, 30, 18, 36, 27, 32, 5, 36, 15, 30, 35, 20, 68, 75, 15, 25, 16, 24, 12, 19, 17, 65, 100, 35, 7, 75, 20, 25}, {40, 63, 60, 37, 26, 41, 13, 19, 19, 26, 20, 12, 18, 53, 100, 30, 25, 66, 75, 34, 27, 35, 13, 24, 12, 22, 20, 7, 13, 21, 40, 10, 38, 62, 62, 38, 20, 45, 11, 24, 26, 36, 24, 14, 12, 27, 48, 14, 13, 65, 8, 35, 15, 26, 6, 15, 79, 61, 100, 39, 6, 62, 10, 38}, {5, 96, 95, 4, 3, 65, 31, 2, 1, 25, 4, 3, 2, 96, 100, 2, 5, 93, 95, 7, 2, 41, 54, 2, 3, 47, 38, 7, 0, 36, 60, 2, 15, 95, 85, 5, 4, 57, 36, 2, 8, 64, 19, 10, 3, 55, 41, 2, 3, 95, 17, 5, 2, 41, 28, 1, 80, 91, 100, 9, 0, 98, 2, 2}, {47, 52, 53, 48, 25, 24, 11, 40, 47, 9, 16, 15, 18, 31, 100, 51, 62, 46, 38, 54, 40, 16, 10, 35, 11, 5, 5, 16, 12, 8, 16, 17, 58, 41, 42, 59, 24, 25, 6, 45, 32, 14, 10, 43, 16, 23, 20, 42, 53, 55, 29, 45, 19, 20, 9, 29, 18, 35, 100, 65, 15, 44, 33, 56}, {39, 55, 61, 45, 29, 37, 8, 26, 19, 22, 19, 15, 14, 51, 100, 35, 24, 56, 76, 44, 29, 32, 8, 31, 14, 18, 19, 11, 8, 19, 39, 13, 41, 53, 59, 47, 23, 37, 7, 32, 25, 34, 21, 20, 12, 24, 46, 18, 26, 56, 22, 44, 14, 22, 5, 22, 53, 53, 100, 47, 7, 53, 14, 47}, {49, 40, 51, 60, 33, 19, 9, 39, 32, 13, 26, 17, 25, 25, 100, 50, 58, 39, 42, 61, 40, 13, 10, 37, 11, 8, 8, 15, 10, 14, 12, 26, 55, 32, 45, 68, 31, 17, 8, 44, 34, 17, 15, 34, 17, 17, 25, 41, 41, 43, 19, 57, 23, 14, 7, 26, 41, 43, 100, 57, 14, 42, 24, 58}, {30, 60, 70, 40, 22, 33, 21, 24, 16, 21, 25, 11, 20, 47, 100, 33, 39, 57, 61, 43, 26, 24, 26, 24, 9, 23, 15, 12, 8, 25, 25, 18, 36, 56, 64, 44, 19, 33, 23, 25, 20, 39, 21, 21, 11, 29, 36, 24, 24, 63, 32, 37, 15, 22, 14, 17, 44, 66, 100, 34, 8, 63, 15, 37}};
        letterCode = LetterCode.e1_Letter_Code;
        mappedOrganisms = new HashMap();
        organisms = new Organism[orgNames.length];
        block5: for (int x = 0; x < orgNames.length; ++x) {
            switch (x) {
                case 6: {
                    CodonTable.organisms[x] = CodonTable.addOrganism(orgNames[x], frequencies[x], 1, x);
                    continue block5;
                }
                case 7: {
                    CodonTable.organisms[x] = CodonTable.addOrganism(orgNames[x], frequencies[x], 2, x);
                    continue block5;
                }
                case 11: {
                    CodonTable.organisms[x] = CodonTable.addOrganism(orgNames[x], frequencies[x], 3, x);
                    continue block5;
                }
                default: {
                    CodonTable.organisms[x] = CodonTable.addOrganism(orgNames[x], frequencies[x], 0, x);
                }
            }
        }
        organismCount = orgNames.length;
    }

    private static class Organism {
        int index = 0;
        String name = "";
        int[] frequencies;
        char[] cAA;
        String[] A;
        String[] AA;
        String[] AAA;
        boolean isDefault = false;

        private Organism() {
        }

        public Organism clone() {
            Organism clone = new Organism();
            clone.index = this.index;
            clone.name = this.name;
            clone.frequencies = this.frequencies;
            clone.cAA = this.cAA;
            clone.A = this.A;
            clone.AA = this.AA;
            clone.AAA = this.AAA;
            clone.isDefault = this.isDefault;
            return clone;
        }
    }

    public static enum LetterCode {
        e1_Letter_Code,
        eSpaced_1_Letter_Code,
        e3_Letter_Code;

    }
}

